package cn.weipass.pos.sdk;

import android.nfc.tech.IsoDep;

import cn.shellinfo.wall.remote.ParamMap;
import cn.weipass.service.operation.IOperationCallback;

/**
 * 运营业务操作类
 * Created by hedong on 2017/3/22.
 */

public interface OperationManager extends Initializer{
    /**
     * 默认值
     */
    public static final int RESULT_NONE = 0;
    /**
     * 处理成功
     */
    public static final int RESULT_SUCCESS = 1;
    /**
     * 一般错误，可能是网络连接超时
     */
    public static final int RESULT_ERR_NORMAL = 1000;
    /**
     * 无网络连接
     */
    public static final int RESULT_ERR_NOCONNECT = 1001;
    /**
     * 连接重定向
     */
    public static final int RESULT_ERR_REDIRECTION = 1002;
    /**
     * 无法连接旺UI的服务器
     */
    public static final int RESULT_ERR_CANNOT_CONNECT_SERVER = 1003;
    /**
     * 无法解析旺UI服务器下行的数据，一般是客户端与服务器端的sessionKey不一致导致的
     */
    public static final int RESULT_ERR_INVAILD_DATA = 1004;
    /**
     * 业务接口返回参数key，如果有此key就表示出错了
     */
    public static final String RESP_KEY_ERROR = "error";

    /**
     * 获得管理模块类实例，例如设备管理、应用管理等
     * @param key 管理类
     * @return 管理类实例
     * @see DeviceManager
     * @see AppManager
     * @see MerchantManager
     * @see SubscribeManager
     * @see UpgradeManager
     * @see MobileManager
     * @see IBeaconManager
     * @see AccountManager
     * @see OrderServiceManager
     * @see SimManager
     */
    public <T>T getModule(Class<T> key);

    /**
     * 设备管理
     */
    public interface DeviceManager{
        /**
         * 激活设备的回调接口类
         */
        public interface ActiveDeviceCallback {
            /**
             * 处理失败后的结果回调
             *
             * @param errCode 错误码<br/>
             * {@link OperationManager#RESULT_ERR_CANNOT_CONNECT_SERVER}<br/>
             * {@link OperationManager#RESULT_ERR_INVAILD_DATA}<br/>
             * {@link OperationManager#RESULT_ERR_NOCONNECT}<br/>
             * {@link OperationManager#RESULT_ERR_NORMAL}<br/>
             * {@link OperationManager#RESULT_ERR_REDIRECTION}
             * @param err     错误描述信息
             */
            void onError(int errCode, String err);

            /**
             * 处理成功后的返回结果回调
             */
            void onSuccess();
        }

        /**
         * 设备初始化回调接口
         */
        public interface DeviceInitCallback {
            /**
             * 返回结果的回调
             *
             * @param response 返回数据<br/>
             * errCode 错误码说明：<br/>
             * {@link OperationManager#RESULT_ERR_CANNOT_CONNECT_SERVER}<br/>
             * {@link OperationManager#RESULT_ERR_INVAILD_DATA}<br/>
             * {@link OperationManager#RESULT_ERR_NOCONNECT}<br/>
             * {@link OperationManager#RESULT_ERR_NORMAL}<br/>
             * {@link OperationManager#RESULT_ERR_REDIRECTION}<br/>
             * err     错误描述信息
             */
            void onResult(ParamMap response);
        }

        /**
         * *获取设备信息
         **/
        public String getDeviceInfo();

        /**
         * 激活设备
         *
         * @param callback 结果回调函数
         */
        public void activeDevice(ActiveDeviceCallback callback);

        /**
         * 初始化设备
         *
         * @param callback 结果回调函数
         */
        public void deviceInit(DeviceInitCallback callback);

        /**
         * 设备是否激活（是否出厂）
         */
        public boolean isActivatedDevice();
        /**
         * 是否初始化设备
         */
        public boolean isDeviceInit();
        /**
         * 是否正在做设备初始化的操作
         * @return
         */
        public boolean isDeviceInitializing();

        /**
         * 清空业务数据
         *
         * @param callback 回调对象
         */
        public void cleanBizDatas(OperationCallback callback);

        /**
         * 验证权限卡
         *
         * @param cardId    nfc卡卡id
         * @param isoDep nfc tag
         * @param callback 回调对象
         */
        public void authRoleCard(String cardId, IsoDep isoDep, OperationCallback callback);

        /**
         * 根据设备SN号查询设备信息
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getDeviceBySn(ParamMap param, OperationCallback callback);

        /**
         * 设置adb是否开启的状态（重启设备后将会恢复到默认配置）
         * @param isOpen 是否开启
         */
        public void setAdbState(boolean isOpen);

        /**
         * 是否打开了adb
         * @return
         */
        public boolean isOpenAdb();

        /**
         * 根据模块获取设备配置
         */
        public void getDeviceConfig(ParamMap param,OperationCallback listener) ;

        public void setSpeechState(boolean isOpen);

        public boolean getSpeechState();
    }


// 业务请求API 开始====================>>

    /**
     * 业务接口回调对象
     */
    public interface OperationCallback {
        /**
         * 调用返回
         *
         * @param response 返回参数，如果response内包含error字段就表示调用出错了。
         */
        public void onResult(ParamMap response);
    }

    /**
     * 应用管理
     */
    public interface AppManager{
        /**
         * 获取全部应用信息(wui3.0起使用)
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getAllAppsInfo(ParamMap param, OperationCallback callback);

        /**
         * 获取全部应用列表
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getAllAppList(ParamMap param, OperationCallback callback);

        /**
         * 删除业务绑定关系
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void removeAuth(ParamMap param, OperationCallback callback);

        /**
         * 获取桌面显示列表数据
         *
         * @param callback 回调对象
         */
        public void getDesktopItemData(OperationCallback callback);

        /**
         * 获取主屏定制信息
         *
         * @param callback 回调对象
         */
        public void getCustomDesktopConfigNew(OperationCallback callback);

        /**
         * 上报应用下载完成
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void reportDownload(ParamMap param, OperationCallback callback);

        /**
         * 获取最新推送的测试应用信息
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getLastPushBizApp(ParamMap param, OperationCallback callback);

        /**
         * 绑定基础业务
         *
         * @param callback 回调对象
         */
        public void baseBizApp(OperationCallback callback);

        /**
         * 获取测试应用列表
         *
         * @param callback 回调对象
         */
        public void getTestApp(OperationCallback callback);
    }

    /**
     * 商户管理
     */
    public  interface MerchantManager{
        /**
         * 绑定商家
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void bindMerchant(ParamMap param, OperationCallback callback);

        /**
         * 无密码商户绑定
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void greenLogin(ParamMap param, OperationCallback callback);

        /**
         * 上传店铺信息
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void uploadMerchantInfo(ParamMap param, OperationCallback callback);

        /**
         * 解除设备绑定商家
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void unbindMerchant(ParamMap param, OperationCallback callback);
        /**
         * 设备店铺业务迁移
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void moveMerchant(ParamMap param, OperationCallback callback);

        /**
         * 店铺安全保护
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void operSafeModel(ParamMap param, OperationCallback callback);

        /**
         * 是否绑定店铺
         * @return
         */
        public boolean isBindMerchant();

        /**
         * 获取商户信息
         * @return
         */
        public ParamMap getMerchantInfo();
    }



    /**
     * 账户管理
     */
    public interface AccountManager{
        /**
         * 发送短信验证码
         * 此接口只适用于员工卡管理相关接口
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void sendVerifyCode(ParamMap param, OperationCallback callback);
        /**
         * 修改员工卡账号密码
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void updateAPsd(ParamMap param, OperationCallback callback);

        /**
         * 员工卡绑定
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void bindStaff(ParamMap param, OperationCallback callback);

        /**
         * 员工卡注销
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void offCard(ParamMap param, OperationCallback callback);

        /**
         * 员工卡解绑
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void unbindStaff(ParamMap param, OperationCallback callback);

        /**
         * 获取员工卡列表
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getStaffCardList(ParamMap param, OperationCallback callback);

        /**
         * 添加员工卡
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void addStaffCard(ParamMap param, OperationCallback callback);

        /**
         * 员工登录店铺
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void staffLogin(ParamMap param, OperationCallback callback);

        /**
         * 添加员工
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void addStaff(ParamMap param, OperationCallback callback);
        /**
         * 获取所有员工信息列表
         * @param callback 回调对象
         */
        public void listStaff(OperationCallback callback);
        /**
         * 修改员工信息
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void editStaff(ParamMap param, OperationCallback callback);
        /**
         * 删除员工信息
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void removeStaff(ParamMap param, OperationCallback callback);
        /**
         * 设置员工为超级管理员
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void setSuperStaff(ParamMap param, OperationCallback callback);
        /**
         * 获取超级管理员（员工）信息
         * @param callback 回调对象
         */
        public void getSuperStaff(OperationCallback callback);
        /**
         * 员工权限验证
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void postAuthInfo(ParamMap param, OperationCallback callback);
        /**
         * 获取员工权限
         * @param callback 回调对象
         */
        public void getAuthInfo(OperationCallback callback);
        /**
         * 上报员工操作日志
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void postOperationLog(ParamMap param, OperationCallback callback);
        /**
         * 获取超级管理员验证码，找回密码时使用
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getStaffVerifyCode(ParamMap param, OperationCallback callback);
        /**
         * 修改管理员密码
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void modifyStaffPassword(ParamMap param, OperationCallback callback);
        /**
         * 获取手机短信验证码,绑定手机时使用
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getVerfiyCode(ParamMap param, OperationCallback callback);
    }

    /**
     * 订单管理
     */
    public interface SubscribeManager{
        /**
         * 申请订阅服务
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void bizAuthApply(ParamMap param, OperationCallback callback);
        /**
         * 处理订单
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void dealSubscribeMsg(ParamMap param, OperationCallback callback);

        /**
         * 获取设备可订阅服务列表
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void searchSubscribeApp(ParamMap param, OperationCallback callback);

        /**
         * 获取已订阅服务列表
         *
         * @param callback 回调对象
         */
        public void getSubscribeList(OperationCallback callback);

        /**
         * 获取已订阅服务中消息模板内容
         *
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getSubscribeMsgList(ParamMap param, OperationCallback callback);
    }

    /**
     * 更新管理
     */
    public interface UpgradeManager{
        /**
         * 准备检查是否有新版本
         */
        public static final int STATUS_TOCHECK = 0;
        /**
         * 正在检查是否有新版本
         */
        public static final int STATUS_CHECKING = 10;
        /**
         * 无新版本
         */
        public static final int STATUS_NO_NEW_VERSION = 20;
        /**
         * 有新的ota升级包
         */
        public static final int STATUS_FOUND_NEW_VERSION_OTA = 30;
        /**
         * 有新的旺UI升级包
         */
        public static final int STATUS_FOUND_NEW_VERSION_WUI = 40;
        /**
         * 正在下载ota升级包
         */
        public static final int STATUS_DOWNLOADING_OTA = 50;
        /**
         * 正在下载旺UI升级包
         */
        public static final int STATUS_DOWNLOADING_WUI = 60;
        /**
         * 下载完成OTA升级包
         */
        public static final int STATUS_DOWNLOADED_OTA = 70;
        /**
         * 下载完成旺UI升级包
         */
        public static final int STATUS_DOWNLOADED_WUI = 80;

        /**
         * 没有新版本，或者本地已经是最新版本
         */
        static final int ERR_NEWEST = 0;
        /**
         * 拉取ota版本信息失败
         */
        static final int ERR_GET_VERSION_OTA = 1;
        /**
         * 拉取旺UI版本信息失败
         */
        static final int ERR_GET_VERSION_WUI = 2;
        /**
         * 下载系统更新包失败
         */
        static final int ERR_DOWNLOAD = 3;
        /**
         * 系统更新包文件完整性校验失败，文件损坏
         */
        static final int ERR_PACKAGE_INTEGRITY = 4;
        /**
         * 系统更新包签名校验失败
         */
        static final int ERR_PACKAGE_SIGN = 5;
        /**
         * 电量过低
         */
        static final int ERR_LOW_BATTERY = 6;
        /**
         * 下载不支持2g环境
         */
        static final int ERR_NOTSUPPORT_2G = 7;
        /**
         * 无网络连接
         */
        static final int ERR_NO_NETWORK = 8;
        /**
         * 磁盘空间不够
         */
        static final int ERR_NOTENOUGH_DISK = 9;
        /**
         * 获取系统版本信息
         * @return 版本信息
         */
        public ParamMap getSystemVersion();

        /**
         * 系统升级监听回调对象
         */
        public interface SystemUpgradeCallback {
            /**
             * 状态更新回调
             * @param status 状态码</br>
             *               {@link UpgradeManager#STATUS_TOCHECK}<br/>
             *               {@link UpgradeManager#STATUS_CHECKING}<br/>
             *               {@link UpgradeManager#STATUS_NO_NEW_VERSION}<br/>
             *               {@link UpgradeManager#STATUS_FOUND_NEW_VERSION_OTA}<br/>
             *               {@link UpgradeManager#STATUS_FOUND_NEW_VERSION_WUI}<br/>
             *               {@link UpgradeManager#STATUS_DOWNLOADING_OTA}<br/>
             *               {@link UpgradeManager#STATUS_DOWNLOADING_WUI}<br/>
             *               {@link UpgradeManager#STATUS_DOWNLOADED_OTA}<br/>
             *               {@link UpgradeManager#STATUS_DOWNLOADED_WUI}<br/>
             * @param paramMap 包含的字段說明：
             *                 参数字段	类型	说明
             *                 version	String	版本名称
             *                 versionNumber	Int	版本号
             *                 desc	String	升级包描述
             *                 type	int	升级类型
             *                 1：ota
             *                 2：旺UI
             *                 fileSize	long	文件大小
             */
            void onChangeStatus(int status,ParamMap paramMap);
            /**
             * 发生错误时的回调
             *
             * @param errorCode 错误码<br/>
             *                  {@link UpgradeManager#ERR_DOWNLOAD}<br/>
             *                  {@link UpgradeManager#ERR_GET_VERSION_OTA}<br/>
             *                  {@link UpgradeManager#ERR_GET_VERSION_WUI}<br/>
             *                  {@link UpgradeManager#ERR_LOW_BATTERY}<br/>
             *                  {@link UpgradeManager#ERR_NEWEST}<br/>
             *                  {@link UpgradeManager#ERR_NO_NETWORK}<br/>
             *                  {@link UpgradeManager#ERR_NOTENOUGH_DISK}<br/>
             *                  {@link UpgradeManager#ERR_NOTSUPPORT_2G}<br/>
             *                  {@link UpgradeManager#ERR_PACKAGE_SIGN}
             * @param error     错误描述
             */
            void onError(int errorCode, String error);
        }

        /**
         * 注册检测系统升级更新的回调对象
         *
         * @param callback 回调对象
         */
        public void regSystemUpgradeCallback(SystemUpgradeCallback callback);

        /**
         * 进入检测系统更新的界面
         */
        public void entryUpgradePanel();

        /**
         * 检测系统更新
         */
        public void checkSystemUpgrade();
        /**
         * 开始下载系统更新包
         */
        public void startDownloadPackage();
        /**
         * 退出检测系统更新的界面
         */
        public void leaveUpgradePanel();

        /**
         * 确认更新升级包（下载完更新包后才有效）
         */
        public void confirmUpdateSystem();
        /**
         * 是否可以执行系统升级安装包（下载完更新包后才有效）
         */
        public boolean canUpgradeSystem();
        /**
         * 刷新升级状态,可能会触发UI更新
         */
        public void freshUpgradeStatus();

        /**
         * 是否已经有下载完的ota升级包
         * @return true：表示已经下周完成，可以执行安装升级<br/>
         * false：没有升级包
         */
        public boolean isOTAPackageReady();
        /**
         * 是否已经有下载完的旺UI升级包
         * @return true：表示已经下周完成，可以执行安装升级<br/>
         * false：没有升级包
         */
        public boolean isWUIPackageReady();
        /**
         * 是否已经有下载完的系统应用升级包
         * @return true：表示已经下周完成，可以执行安装升级<br/>
         * false：没有升级包
         */
        public boolean isSysAppPackageReady();
    }

    /**
     * 获取指定尺寸的图片
     *
     * @param param    用于保存业务请求参数的ParamMap对象
     * @param callback 回调对象
     */
    public void getImg(ParamMap param, OperationCallback callback);

    /**
     * 移动网络管理
     */
    public interface MobileManager{
        /**
         * 获取sim卡的余额、套餐信息
         *
         * @param callback 回调对象
         */
        public void getSimInfo(OperationCallback callback);

        /**
         * 设置手机号码信息
         *
         * @param param
         */
        public void setMobileInfo(ParamMap param);
        /**
         * 获取手机号
         * @return
         */
        public String getSimMobile();
        /**
         * 获取手机序列号
         * @return
         */
        public String getSimSerialNumber();
        /**
         * 设置自定义apn
         *
         * @param paramMap
         */
        public void setCustomApn(ParamMap paramMap);
        /**
         * 获取自定义apn设置参数
         * @return
         */
        public String getCustomApn();

    }


    /**
     * 上报意见反馈
     * @param param    用于保存业务请求参数的ParamMap对象
     * @param callback 回调对象
     */
    public void reportFeed(ParamMap param, OperationCallback callback);

    /**
     * Ibeacon管理
     */
    public interface IBeaconManager{
        /**
         * 绑定IBeacon
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void bindBeacon(ParamMap param, OperationCallback callback);
        /**
         * IBeacon获取随机数
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getBeaconRandomData(ParamMap param, OperationCallback callback);
        /**
         * 查询IBeacon设备绑定关系
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getBeaconList(ParamMap param, OperationCallback callback);
        /**
         * 获取IBeacon设备广播的内容
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void getBeanconContent(ParamMap param, OperationCallback callback);
        /**
         * 刷新IBeacon数据
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void refreshBeaconData(ParamMap param, OperationCallback callback);
        /**
         * 上传IBeacon资讯数据
         * @param param    用于保存业务请求参数的ParamMap对象
         * @param callback 回调对象
         */
        public void setBeanconContent(ParamMap param, OperationCallback callback);
    }

//  业务请求API 结束==================>>


//=============API定义100+ start =============

    interface AppMarketManager{

        /**
         * 查询推荐列表下面的数据
         * @param callback
         */
        void getRecommendAppList(OperationCallback callback);

        /**
         * 查询类型分类
         * @param callback
         */
        void getTypeClassifyList(OperationCallback callback);

        /**
         * 查询某个类型的APP列表
         * @param callback
         */
        void queryTypeAppList( ParamMap param,OperationCallback callback);

        /**
         * 获取我的里面列表数据
         * @param param
         * @param callback
         */
        void queryMineAppList(ParamMap param, OperationCallback callback);

        /**
         * 上报信息
         * @param param
         * @param callback
         */
        void reportFeed(ParamMap param, OperationCallback callback);

        /**
         * 上报信息
         * @param param
         * @param callback
         */
        void bizAuthApply(ParamMap param, OperationCallback callback);
    }


    //=============API定义100+ end =============

    //=============API定义200+ start =============
    /**
     * 订单服务相关
     */
    interface OrderServiceManager{

        /**
         * 获取设备下某个服务的服务调用消息记录-----新
         * @param param
         * @param callback
         */
        void getServiceSeqList(ParamMap param, OperationCallback callback);

        /**
         * 获取设备已订阅、可订阅服务列表-----新
         * @param callback
         */
        void getSubscribeServiceNewList(OperationCallback callback);

        /**
         * 解绑已订阅服务-----新
         * @param param
         * @param callback
         */
        void unSubscribeService(ParamMap param, OperationCallback callback);

        /**
         * 推送 服务调用消息-------新
         * @param param
         * @param callback
         */
        void serviceInvokerCallback(ParamMap param, OperationCallback callback);
    }

    /**
     * SIM卡管理
     */
    interface SimManager{
        /**
         * 检测sim卡绑定状态
         * @param callback
         */
        void checkSimBind(OperationCallback callback);

        /**获取历史订单
         * @param paramMap
         * @param listener
         */
        void getCardBillList(ParamMap paramMap ,OperationCallback listener);

        /**是否是物联网卡
         * @param paramMap
         * @param listener
         */
        void isInSimDB(ParamMap paramMap ,OperationCallback listener);

        /**获取未支付订单
         * @param paramMap
         * @param listener
         */
        void getUnPaidMoneyByEn(ParamMap paramMap ,OperationCallback listener) ;

        /**获取定制SIM布局
         * @param paramMap
         * @param listener
         */
        void getPosDisplayInfo(ParamMap paramMap ,OperationCallback listener);

        /**取消未支付订单
         * @param paramMap
         * @param listener
         */
        void cancelUnPaidWxBill(ParamMap paramMap ,OperationCallback listener);

        /**获取SIM基础信息
         * @param paramMap
         * @param listener
         */
        void getBaseSimInfo(ParamMap paramMap ,OperationCallback listener);

        /**进行账单支付
         * @param paramMap
         * @param listener
         */
        void billPurchase(ParamMap paramMap ,OperationCallback listener);

        /**綁定SIM
         * @param paramMap
         * @param callback
         */
        void BindSimToPos(ParamMap paramMap,OperationCallback callback);

        /**上传流量排行到后台
         * @param paramMap
         * @param callback
         */
        void addFlowDetail(ParamMap paramMap,OperationCallback callback);
    }
    //=============API定义200+ end =============
}
