package cn.weipass.pos.sdk;

import android.app.Notification;
import android.content.Context;

import java.util.List;

import cn.weipass.pos.sdk.execption.DeviceStatusException;

/**
 * 
 * 微POS设备实例接口类，可以基于该实例获取微POS设备状态、打开微POS的各种外设进行交互和通讯。可以通过以下的方式获得微POS设备实例：
 * 
 * <p/>
 * 
 * <pre>
 * Weipos weipos = WeiposImpl.as();
 * </pre>
 * 
 * @since 1.0
 * @author Happy
 * 
 */
public interface Weipos {

	/**
	 * 电源状态，值包括：1-外部电源供电，2-电池供电，3-正在充电
	 */
	public static final int STATUS_POWER = 1;
	/**
	 * 当前网络状态，值包括：1-gprs，2-3g，3-wifi/网线
	 */
	public static final int STATUS_NETWORK = 2;
	/**
	 * 当前电池电量，值为0-100的数值
	 */
	public static final int STATUS_BATTERY = 3;

	/**
	 * 微POS初始化工作监听类
	 * 
	 * @since 1.0
	 * @author hedong
	 * 
	 */
	public interface OnInitListener {
		/**
		 * 成功完成初始化后的回调方法
		 */
		public void onInitOk();

		/**
		 * SDK初始化、获取旺POS服务对象失败时的回调方法
		 * 
		 * @param msg
		 *            失败消息
		 */
		public void onError(String msg);
		
		/**
		 * 旺pos服务对象销毁时回调
		 */
		public void onDestroy();
	}

	/**
	 * 取得微POS的各种状态值<br>
	 * 此函数已过时，不能使用。
	 * @deprecated
	 * @see #STATUS_POWER
	 * @see #STATUS_NETWORK
	 * @see #STATUS_BATTERY
	 * 
	 * @param field
	 *            状态值字段
	 * @return 状态值
	 */
	public int getWeiposStatus(int field);

	/**
	 * 微POS初始化方法，建议在Android Activity onCreate方法中调用
	 * 
	 * @param context
	 *            Android Context
	 * @param listener
	 *            初始化监听类{@link OnInitListener}
	 * @see OnInitListener
	 */
	public void init(Context context, OnInitListener listener);

	/**
	 * 微POS资源释放方法，建议在Android Activity onDestroy方法中调用
	 */
	public void destroy();

	/**
	 * 设置初始化监听类
	 * @param listener 初始化监听类{@link OnInitListener}
	 * @see OnInitListener
	 */
	public void setOnInitListener(OnInitListener listener);

	/**
	 * 获取一般打印服务对象<br>
	 * 此函数已过时，建议使用 {@link #getService}(Printer.class)来获取一般打印服务对象<br>
	 * 为了更好的提供打印服务，例如控制打印内容排版，
	 * 建议使用{@link LatticePrinter} 点阵打印服务对象来完成打印工作，而{@link Printer}类将可能不会继续维护更新
	 * @see Printer
	 * @see LatticePrinter
	 * @deprecated
	 * @return 打印设备对象
	 * @throws DeviceStatusException
	 */
	public Printer openPrinter() throws DeviceStatusException;
	/**
	 * 获取点阵打印服务对象。<br>
	 * 建议使用 {@link #getService}(LatticePrinter.class)来获取点阵打印机服务对象
	 * @see LatticePrinter
	 * @return 点阵打印服务对象
	 * @throws DeviceStatusException
	 */
	public LatticePrinter openLatticePrinter() throws DeviceStatusException;

	/**
	 * 获取光学扫描服务对象，提供扫描二维码、条形码服务。<br>
	 * 建议使用 {@link #getService}(Scanner.class)来获取光学扫描服务对象
	 * @see Scanner
	 * @return 光学扫描服务对象
	 * @throws DeviceStatusException
	 */
	public Scanner openScanner() throws DeviceStatusException;

	/**
	 * 打开旺POS平台数据通讯服务对象<br>
	 * 建议使用 {@link #getService}(DataChannel.class)来获取旺POS平台数据通讯服务对象
	 * @see DataChannel
	 * @return 旺POS平台数据通讯服务对象
	 * @throws DeviceStatusException
	 */
	public DataChannel openDataChannel() throws DeviceStatusException;

	/**
	 * 打开声波近场通讯服务对象<br>
	 * 建议使用 {@link #getService}(Sonar.class)来获取声波近场通讯服务对象
	 * @see Sonar
	 * @return 声波近场通讯服务对象
	 * @throws DeviceStatusException
	 */
	public Sonar openSonar() throws DeviceStatusException;
	/**
	 * 打开业务服务提供者对象<br>
	 * 建议使用 {@link #getService}(BizServiceProvider.class)来获取业务服务提供者对象
	 * @see BizServiceProvider
	 * @return 业务服务提供者对象
	 * @throws DeviceStatusException
	 */
	public BizServiceProvider openBizServiceProvider() throws DeviceStatusException;
	/**
	 * 打开业务服务调用者对象<br>
	 * 建议使用 {@link #getService}(BizServiceInvoker.class)来获取业务服务调用者对象
	 * @see BizServiceInvoker
	 * 
	 * @return 业务服务调用者对象
	 * @throws DeviceStatusException
	 */
	public BizServiceInvoker openBizServiceInvoker() throws DeviceStatusException;

	/**
	 * 打开拍照服务对象<br>
	 * 建议使用 {@link #getService}(Photograph.class)来获取拍照服务对象
	 * @see Photograph
	 * @return 拍照服务对象
	 * @throws DeviceStatusException
	 */
	public Photograph openPhotograph() throws DeviceStatusException;

	/**
	 * 打开磁卡读卡器服务对象<br>
	 * 建议使用 {@link #getService}(MagneticReader.class)来获取磁卡读卡器服务对象
	 * @return 磁卡读卡器服务对象
	 * @throws DeviceStatusException
	 */
	public MagneticReader openMagneticReader() throws DeviceStatusException;

	/**
	 * 打开操作员授权管理器服务对象<br>
	 * 建议使用 {@link #getService}(AuthorizationManager.class)来获取操作员授权管理器服务对象
	 * @return 操作员授权管理器服务对象
	 * @throws DeviceStatusException
	 */
	public AuthorizationManager openAuthorizationManager() throws DeviceStatusException;

	/**
	 * 打开基础服务管理器对象<br>
	 * 建议使用 {@link #getService}(ServiceManager.class)来获取基础服务管理器对象
	 * @return 基础服务管理器对象
	 * @throws DeviceStatusException
	 */
	public ServiceManager openServiceManager() throws DeviceStatusException;

	/**
	 * 获取设备信息
	 * @return 返回json格式的数据。<br>具体包含的key为：<br>
	 * mname:商户名称<br>
	 * mcode：商户Code<br>
	 * deviceType:设备类型，2,2s，3<br>
	 * name：登陆用户名<br>
	 * snCode：设备SN号<br>
	 * en：设备EN号<br>
	 * loginType：员工卡类型
	 **/
	public String getDeviceInfo() throws DeviceStatusException;

	/**
	 * 打开Psam服务管理器<br>
	 * 建议使用 {@link #getService}(PsamManager.class)来获取Psam服务管理器
	 * @return Psam服务管理器
	 * @throws DeviceStatusException
	 */
	public PsamManager openPsamManager() throws DeviceStatusException;

	/**
	 * 打开led灯服务管理器<br>
	 * 建议使用 {@link #getService}(LedLightManager.class)来获取led灯服务管理器
	 * @return led灯服务管理器
	 * @throws DeviceStatusException
	 */
	public LedLightManager openLedLightManager() throws DeviceStatusException;

	/**
	 * 打开mini灯光服务管理器<br>
	 * 建议使用 {@link #getService}(MiniLightManager.class)来获取mini灯光服务管理器
	 * @return mini灯光服务管理器
	 * @throws DeviceStatusException
	 */
	public MiniLightManager openMiniLightManager() throws DeviceStatusException;

	/**
	 * Post a notification to be shown in the status bar. If a notification with
	 * the same id has already been posted by your application and has not yet
	 * been canceled, it will be replaced by the updated information.
	 * 
	 * @param id
	 *            An identifier for this notification unique within your
	 *            application.
	 * @param notification
	 *            A Notification object describing what to show the user. Must
	 *            not be null.
	 * @throws DeviceStatusException
	 */
	public void notify(int id, Notification notification) throws DeviceStatusException;

	/**
	 * 根据文字内容播放语音
	 * 
	 * @param content
	 *            要播放的内容
	 */
	public void speech(String content);

	/**
	 * 设置悬浮视图是否可见
	 * @param isVisible 是否可见
	 */
	public void setSuspendViewVisibility(boolean isVisible);
	
	/**
	**获取支持的服务名列表
	**/
	public List<String> getSupportServiceList();
	
	/**
	 * @deprecated
	 * 打开服务对象,通过{@link #getSupportServiceList()}获取到支持服务名称
	 * @param name 服务名
	 * @see #getSupportServiceList()
	 * @return 服务对象
	 */
	public Object openService(String name);
	/**
	 * 打开ped调用对象<br>
	 * 建议使用 {@link #getService}(Ped.class)来获取ped调用对象
	 * @return ped对象
	 */
	public Ped openPed();
	/**
	 * 打开蜂鸣器对象<br>
	 * 建议使用 {@link #getService}(BeepManager.class)来获取BeepManager对象
	 * @return BeepManager对象
	 */
	public BeepManager openBeepManager();
	
	public TabUIManager openTabUIManager();

	public PiccManager openPiccManager();

	public NetworkAddressManager openNetworkAddressManager();
	
	/**
	 * 获取服务对象（服务类定义在SDK中的cn.weipass.pos.adk包下），例如蜂鸣器对象{@link BeepManager}、点阵打印服务对象{@link LatticePrinter}等。<br>
	 * 调用方式例如：LatticePrinter mLatticePrinter=WeiposImp.as().getService(LatticePrinter.class)
	 * @param service 服务类Class对象<br>
	 * @return 服务对象实例
	 * @see DataChannel
	 * @see Scanner
	 * @see Photograph
	 * @see Printer
	 * @see LatticePrinter
	 * @see MagneticReader
	 * @see PsamManager
	 * @see ServiceManager
	 * @see BizServiceInvoker
	 * @see Sonar
	 * @see AuthorizationManager
	 * @see LedLightManager
	 * @see BeepManager
	 * @see BizServiceProvider
	 * @see RSAManager
	 */
	public <T> T getService(Class<?> service);
	/**
	 * 获取运营业务模块对象实例
	 * @param module 模块对象类。包含：<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.DeviceManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.AccountManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.AppManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.IBeaconManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.MerchantManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.MobileManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.OrderServiceManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.SubscribeManager}<br/>
	 *               {@link cn.weipass.pos.sdk.OperationManager.UpgradeManager}<br/>
	 * @param <T>
	 * @return 模块对象实例
	 */
	public <T> T getOperationModule(Class<?> module);
}
